C      *****************************************************************
C      * Subroutine ForwardElim                                        *
C      * Transforms the matrix, [A] into an upper triangular matrix.   *
C      * This is done by ordering the array such that the row with the *
C      * largest coeff. in col 1 is row 1, the row (of the remaining   *
C      * rows) with the largest coefficient in col 2 is row 2, etc.    *
C      * This order, the coefficients, and a linear combination of the *
C      * rows are used to eliminate coefficients to create the         *
C      * triangular matrix.  Note that the matrix [A] is NOT physically*
C      * reordered, the row order is saved in the Order vector.        *
C      * However, the values in matrix [A] are changed in PivotMatrix. *
C      *                                                               *
C      * Variables:                                                    *
C      * A         -In/Out- A matrix of size "Size" by ("Size"+1).     *
C      *                    The "Size" by "Size" portion of the matrix *
C      *                    contains the coefficents of "Size"         *
C      *                    equations in "Size" unknowns, and the      *
C      *                    "Size"+1st column is the right hand side   *
C      *                    vector of constant terms.  "Size" is equal *
C      *                    to the number of components in the system. *
C      *                    Units are consistent with inputs.          *
C      *                    (Common block VGauss, file VGauss.f).      *
C      * Col       -Local - Column index into matrix A and vector X.   *
C      *                    Unitless.                                  *
C      * CrntRow   -Local - Row index for Pth row in the upper         *
C      *                    triangular A matrix.  Unitless.            *
C      * Fctr      -Local - Multiplication factor used with CrntRow so *
C      *                    that when it is subtracted from NextRow a  *
C      *                    coefficient in matrix A is zeroed out.     *
C      *                    Units are consistent with equations.       *
C      * Iter      -Input - The iteration number.  Unitless.           *
C      * NextRow   -Local - Row index for the (P+1)st row in the upper *
C      *                    triangular A matrix.  Unitless.            *
C      * Order     -Input - A vector of size "Size" which orders the   *
C      *                    rows of A so that in this order A is an    *
C      *                    upper triangular matrix.  Unitless.        *
C      *                    (Common block VGauss, file VGauss.f).      *
C      * P         -Local - Row index for upper triangular matrix A.   *
C      *                    Unitless.                                  *
C      * Row       -Local - Row index into matrix A.  Unitless.        *
C      * Size      -Input - The number of chemical components in the   *
C      *                    system.  Unitless.                         *
C      *                    (Common block VGauss, file VGauss.f).      *
C      * UnitNum   -Input - The output file number.  Unitless.         *
C      *****************************************************************
       SUBROUTINE FORWARDELIM(ITER, UNITNUM)
							IMPLICIT NONE
							INCLUDE 'VGAUSS.INC'
							INTEGER P, ROW, COL, NEXTROW, CRNTROW
       INTEGER ITER, UNITNUM
							REAL*8 FCTR
	   
C      *---------------------------------------------------------------*
C      * Find the row numbers with the largest coefficient for each    *
C      * column starting from row 1 to row (Size-1).  These row numbers*
C      * are stored in the Order array where order(P) contains the row *
C      * number with the largest coeff. for col. P, order(P+1) contains*
C      * the row number with the largest coeff. for col P+1 of all rows*
C      * excluding the row in order(P), order(P+2) contains the row    *
C      * number with the largest coeff. for col P+2 of all rows excl.  *
C      * the rows in order(P) and order(P+1).                          *
C      *---------------------------------------------------------------*
	      DO 100 P=1,SIZE-1
	         CALL PIVOTMATRIX(P)

C         *------------------------------------------------------------*
C         * For each row after row Order(P) in the Order array,        *
C         * calculate the ratio between the coefficients in the        *
C         * current col P.  Calculate new coefficients for the next    *
C         * row by subtracting this factor times Row's coefficient     *
C         * from NextRows's coefficient.  This is actually             *
C         * subtracting Row times a constant from NextRow              *
C         *------------------------------------------------------------*
      		  DO 110 ROW=P+1, SIZE
      		     CRNTROW = ORDER(P)
         			 NEXTROW = ORDER(ROW)
													FCTR = A(NEXTROW, P) / A(CRNTROW,P)
													DO 120 COL = P, SIZE+1
			             A(NEXTROW,COL) = A(NEXTROW,COL) - 
     >                           FCTR * A(CRNTROW,COL)
  120        CONTINUE
  110     CONTINUE
  100  CONTINUE
  
C      *---------------------------------------------------------------*
C      * Check for singularity. A singular matrix will have a zero     *
C      * element in the diagonal. Because each line in the matrix is   *
C      * not in the order to produce the diagonal, to find the line    *
C      * with the nth diagonal take the order(n) line.  A singular     *
C      * indicates an underdefined system.                             *
C      *---------------------------------------------------------------*
       FCTR = 1
	      DO 200 ROW = 1, SIZE
	         FCTR = FCTR * A(ORDER(ROW),ROW)
  200  CONTINUE
  
       IF (FCTR.EQ.0) THEN
	         WRITE(6,300) ITER
	         WRITE(UNITNUM,300) ITER
  300     FORMAT(' MATRIX IS SINGULAR IN FORWARDELIM',
     >     'iter = ',I3)
       ENDIF

       RETURN
	      END
C      *****************************************************************
C      *                    END SUBROUTINE                             *
C      *****************************************************************
